import { expect } from 'chai';
import { describe } from 'mocha';
import {
  FameContracts,
  deployContractsForTests,
  getSigners,
} from './helpers/deploy-helper';
import { Signers } from './types';
import { Governance } from '../../types';
import { initializeContracts } from './helpers/contracts-initialization-helper';

describe('81 - Governance Contract Mint and Burn Payment Tokens', function () {
  let governanceContract: Governance;
  let contracts: FameContracts;
  let signers: Signers;
  beforeEach(async function () {
    contracts = await deployContractsForTests();
    governanceContract = contracts.governanceContract;
    signers = await getSigners();
    await initializeContracts({ contracts, signers });
  });
  it('admin can mint payment tokens to user address', async function () {
    const amount = 1000;
    await governanceContract
      .connect(signers.admin)
      .mintCoin(signers.signer1.address, 'FDE', amount);

    const balance = await contracts.paymentTokenContract.balanceOf(
      signers.signer1.address,
    );
    expect(balance.toString()).to.equal(amount.toString());
  });

  it('signer1 cannot mint payment tokens: Revert with Ownable: caller is not the owner', async function () {
    const amount = 1000;
    await expect(
      governanceContract
        .connect(signers.signer1)
        .mintCoin(signers.signer1.address, 'FDE', amount),
    ).to.be.revertedWith('LibDiamond: Must be contract owner');
  });

  // no allowance needed becouse we use burn(bypasses allowance) not burnFrom. Admin is owner of the contract and burn method is only accesible by owner.
  it('admin can burn tokens from any address without allowance', async function () {
    const amount = 1000;

    await governanceContract
      .connect(signers.admin)
      .mintCoin(signers.signer1.address, 'FDE', amount);

    const balanceBefore = await contracts.paymentTokenContract.balanceOf(
      signers.signer1.address,
    );

    await governanceContract
      .connect(signers.admin)
      .burnCoin(signers.signer1.address, 'FDE', amount);

    const balanceAfter = await contracts.paymentTokenContract.balanceOf(
      signers.signer1.address,
    );

    expect(balanceAfter).to.equal(balanceBefore.sub(amount));
  });

  it('burn tokens should work after allowance', async function () {
    const amount = 1000;
    await governanceContract
      .connect(signers.admin)
      .mintCoin(signers.signer1.address, 'FDE', amount);

    // Approve payment token contract to spend tokens
    await contracts.paymentTokenContract
      .connect(signers.signer1)
      .approve(governanceContract.address, amount);

    // Burn tokens
    await governanceContract
      .connect(signers.admin)
      .burnCoin(signers.signer1.address, 'FDE', amount);

    const balance = await contracts.paymentTokenContract.balanceOf(
      signers.signer1.address,
    );
    expect(balance.toString()).to.equal('0');
  });

  it('Ensure that burn can be only performed by admin even if other user has allowance for moving tokens', async function () {
    const amount = 1000;
    await governanceContract
      .connect(signers.admin)
      .mintCoin(signers.signer1.address, 'FDE', amount);

    // Approve payment token contract to spend tokens
    await contracts.paymentTokenContract
      .connect(signers.signer1)
      .approve(governanceContract.address, amount);

    // Approve signer1 to spend signers2 contract
    await contracts.paymentTokenContract
      .connect(signers.signer1)
      .approve(signers.signer2.address, amount);

    // Burn tokens
    await expect(
      governanceContract
        .connect(signers.signer2) // not admin
        .burnCoin(signers.signer1.address, 'FDE', amount),
    ).to.be.revertedWith('LibDiamond: Must be contract owner');

    const balance = await contracts.paymentTokenContract.balanceOf(
      signers.signer1.address,
    );
    expect(balance.toString()).to.equal('1000');
  });
});
